<?php

declare(strict_types=1);

namespace Erlage\Photogram\Data\Dtos\Post;

use Erlage\Photogram\Data\Dtos\AbstractDTO;
use Erlage\Photogram\Data\Dtos\Traits\TraitComplaintValidatorDTO;
use Erlage\Photogram\Data\Dtos\Traits\TraitComplaintTransformerDTO;

final class PostDisplayContentDTO extends AbstractDTO
{
    /**
     * to comply with DTO interface
     */
    use TraitComplaintValidatorDTO;
    use TraitComplaintTransformerDTO;

    /**
     * @var PostDisplayContentItemDTO[] 
     */
    private $items;

    private $iteratorKey = 0;

    /*
    |--------------------------------------------------------------------------
    | constructor without promoted properties, helps static analysis
    |--------------------------------------------------------------------------
    */

    /**
     * @param PostDisplayContentItemDTO[] $items 
     */
    public function __construct($items = array())
    {
        $this -> items = $items;
    }

    /*
    |--------------------------------------------------------------------------
    | from json map
    |--------------------------------------------------------------------------
    */

    /**
     * @return static 
     */
    public static function fromJsonMap(array $data)
    {
        /**
         * @var PostDisplayContentItemDTO[]
         */
        $items = array();

        foreach ($data as $dataKey => $dataItem)
        {
            $items[] = PostDisplayContentItemDTO::fromJsonMap($dataItem);
        }

        return new static($items);
    }

    /*
    |--------------------------------------------------------------------------
    | interface
    |--------------------------------------------------------------------------
    */

    /**
     * whether post content contains any item
     * method simply runs count() call on items property and checks whether
     * count returns a non-negative number
     */
    public function isEmpty(): bool
    {
        return $this -> getNumberOfItems() > 0;
    }

    /**
     * get first item from content.
     */
    public function getFirstItem(): PostDisplayContentItemDTO
    {
        return $this -> items[0];
    }

    public function getNumberOfItems(): int
    {
        return \count($this -> items);
    }

    /**
     * add an item to post content.
     * note: content once added cannot be removed for now. we'll introduce
     * new API for this wrapper that'll include selective selects and deletes
     */
    public function add(PostDisplayContentItemDTO $postDisplayContentItemDTO): self
    {
        $this -> items[] = $postDisplayContentItemDTO;

        return $this;
    }

    /*
    |--------------------------------------------------------------------------
    | iterator related
    |--------------------------------------------------------------------------
    */

    public function iteratorPrepare(): void
    {
        $this -> iteratorKey = -1;
    }

    public function iteratorForward(): bool
    {
        if (isset($this -> items[$this -> iteratorKey + 1]))
        {
            $this -> iteratorKey++;

            return true;
        }

        return false;
    }

    public function iteratorCurrentDisplayContentItem(): PostDisplayContentItemDTO
    {
        return $this -> items[$this -> iteratorKey];
    }

    /*
    |--------------------------------------------------------------------------
    | serialization for json_encode
    |--------------------------------------------------------------------------
    */

    public function jsonSerialize()
    {
        return $this -> items;
    }

    /*
    |--------------------------------------------------------------------------
    | implement DTO methods
    |--------------------------------------------------------------------------
    */

    /**
     * @param PostDisplayContentDTO $postDisplayContentDTO
     */
    public static function serialize($postDisplayContentDTO): string
    {
        return \json_encode($postDisplayContentDTO);
    }

    /**
     * @return PostDisplayContentDTO 
     */
    public static function deSerialize(string $postDisplayContentDTO)
    {
        $data = \json_decode($postDisplayContentDTO, true);

        return self::fromJsonMap($data);
    }
}
